/* Types.h -- Basic types
 2008-11-23 : Igor Pavlov : Public domain */

#ifndef __7Z_TYPES_H
#define __7Z_TYPES_H

#include <stddef.h>

#ifdef WIN32
#include <windows.h>
#endif

#define SZ_OK 0

#define SZ_ERROR_DATA 1
#define SZ_ERROR_MEM 2
#define SZ_ERROR_CRC 3
#define SZ_ERROR_UNSUPPORTED 4
#define SZ_ERROR_PARAM 5
#define SZ_ERROR_INPUT_EOF 6
#define SZ_ERROR_OUTPUT_EOF 7
#define SZ_ERROR_READ 8
#define SZ_ERROR_WRITE 9
#define SZ_ERROR_PROGRESS 10
#define SZ_ERROR_FAIL 11
#define SZ_ERROR_THREAD 12

#define SZ_ERROR_ARCHIVE 16
#define SZ_ERROR_NO_ARCHIVE 17

extern const char *LZMA_ERRORS[];

typedef int SRes;

#ifdef _WIN32
typedef DWORD WRes;
#else
typedef int WRes;
#endif

#ifndef RINOK
#define RINOK(x) { int __result__ = (x); if (__result__ != 0) return __result__; }
#endif

typedef unsigned char Byte;
typedef short Int16;
typedef unsigned short UInt16;

#ifdef _LZMA_LZ_UInt32_IS_ULONG
typedef long LZ_Int32;
typedef unsigned long LZ_UInt32;
#else
typedef int LZ_Int32;
typedef unsigned int LZ_UInt32;
#endif

#ifdef _SZ_NO_INT_64

/* define _SZ_NO_INT_64, if your compiler doesn't support 64-bit integers.
 NOTES: Some code will work incorrectly in that case! */

typedef long LZ_Int64;
typedef unsigned long LZ_UInt64;

#else

typedef long long int LZ_Int64;
typedef unsigned long long int LZ_UInt64;

#endif

#ifdef _LZMA_NO_SYSTEM_SIZE_T
typedef LZ_UInt32 SizeT;
#else
typedef size_t SizeT;
#endif

//typedef int Bool;
#define True 1
#define False 0


#ifdef _MSC_VER

#if _MSC_VER >= 1300
#define MY_NO_INLINE __declspec(noinline)
#else
#define MY_NO_INLINE
#endif

#define MY_CDECL __cdecl
#define MY_STD_CALL __stdcall
#define MY_FAST_CALL MY_NO_INLINE __fastcall

#else

#define MY_CDECL
#define MY_STD_CALL
#define MY_FAST_CALL

#endif


/* The following interfaces use first parameter as pointer to structure */

typedef struct
{
	SRes (*Read)(void *p, void *buf, size_t *size);
	/* if (input(*size) != 0 && output(*size) == 0) means end_of_stream.
	 (output(*size) < input(*size)) is allowed */
} ISeqInStream;

/* it can return SZ_ERROR_INPUT_EOF */
SRes SeqInStream_Read(ISeqInStream *stream, void *buf, size_t size);
SRes SeqInStream_Read2(ISeqInStream *stream, void *buf, size_t size, SRes errorType);
SRes SeqInStream_ReadByte(ISeqInStream *stream, Byte *buf);

typedef struct
{
	size_t (*Write)(void *p, const void *buf, size_t size);
	/* Returns: result - the number of actually written bytes.
	 (result < size) means error */
} ISeqOutStream;

typedef enum
{
	SZ_SEEK_SET = 0,
	SZ_SEEK_CUR = 1,
	SZ_SEEK_END = 2
} ESzSeek;

typedef struct
{
	SRes (*Read)(void *p, void *buf, size_t *size);  /* same as ISeqInStream::Read */
	SRes (*Seek)(void *p, LZ_Int64 *pos, ESzSeek origin);
} ISeekInStream;

typedef struct
{
	SRes (*Look)(void *p, void **buf, size_t *size);
	/* if (input(*size) != 0 && output(*size) == 0) means end_of_stream.
	 (output(*size) > input(*size)) is not allowed
	 (output(*size) < input(*size)) is allowed */
	SRes (*Skip)(void *p, size_t offset);
	/* offset must be <= output(*size) of Look */

	SRes (*Read)(void *p, void *buf, size_t *size);
	/* reads directly (without buffer). It's same as ISeqInStream::Read */
	SRes (*Seek)(void *p, LZ_Int64 *pos, ESzSeek origin);
} ILookInStream;

SRes LookInStream_LookRead(ILookInStream *stream, void *buf, size_t *size);
SRes LookInStream_SeekTo(ILookInStream *stream, LZ_UInt64 offset);

/* reads via ILookInStream::Read */
SRes LookInStream_Read2(ILookInStream *stream, void *buf, size_t size, SRes errorType);
SRes LookInStream_Read(ILookInStream *stream, void *buf, size_t size);

#define LookToRead_BUF_SIZE (1 << 14)

typedef struct
{
	ILookInStream s;
	ISeekInStream *realStream;
	size_t pos;
	size_t size;
	Byte buf[LookToRead_BUF_SIZE];
} CLookToRead;

void LookToRead_CreateVTable(CLookToRead *p, int lookahead);
void LookToRead_Init(CLookToRead *p);

typedef struct
{
	ISeqInStream s;
	ILookInStream *realStream;
} CSecToLook;

void SecToLook_CreateVTable(CSecToLook *p);

typedef struct
{
	ISeqInStream s;
	ILookInStream *realStream;
} CSecToRead;

void SecToRead_CreateVTable(CSecToRead *p);

typedef struct
{
	SRes (*Progress)(void *p, LZ_UInt64 inSize, LZ_UInt64 outSize);
	/* Returns: result. (result != SZ_OK) means break.
	 Value (LZ_UInt64)(LZ_Int64)-1 for size means unknown value. */
} ICompressProgress;

typedef struct
{
	void *(*Alloc)(void *p, size_t size);
	void (*Free)(void *p, void *address); /* address can be 0 */
} ISzAlloc;

#define IAlloc_Alloc(p, size) (p)->Alloc((p), size)
#define IAlloc_Free(p, a) (p)->Free((p), a)

#endif
